/*
   dualsearch.cpp
   Supporting material software to the article 
   COUNTING AND CORRECTING THERMODYNAMICALLY INFEASIBLE FLUX CYCLES IN GENOME-SCALE METABOLIC NETWORKS 

   Copyright (C) 2013  D.Demartino, F.Capuani, M.Mori, A.De Martino & E.Marinari

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "iostream"
#include "stdlib.h"
#include "math.h"
#include "time.h"
#include "fstream"
#include "vector"
#include "string"
using namespace std;


//Names of the INPUT files
#define Stoichiometric_matrix_filename  "matrix.dat"	// Name of the file containing the Stoichiometric Matrix (we need M rows and N columns)
#define fluxes_filename  "fluxes.dat"			// Name of the file containing the flux configuration that needs to be corrected. It should contain N elements
#define potentials_filename  "potentials.dat"           // OPTIONAL: name of the file containing the initial chemical potentials. In the maniscript this file was never used and the corresponding part of the code is commented out.
//Size of Stoichiometric matrix
#define N 1774                                             // number of reactions
#define M 1169                                             // number of metabolites
//Name of the OUTPUT files
#define cycles_filename "cycles.dat"                         // Name of file containing the closed cycles identified. Columns are 1- ID of reaction r, k_r, and v_r (flux value of reaction r), respectively
#define feasible_fluxes_filename "new_fluxes.dat"	     // Fluxes after correction has taken place
#define feasible_potentials_filename "new_potentials.dat"    // Chemical potentials that solve Eq. (2)

// ALGORITHM PARAMETERS
#define max_LUC 200           // lenght of the list of l.u.c. (least unsatisfied constraints)
#define max2 20               // number of times the list of l.u.c. is updated before checking for cycles with Montecarlo (MC)       
#define max_MC_sweeps 20      // max number of MC trials before reassign chemical potentials 
double T1 = 1.5;              // initial temperature of the Montecarlo simulated annealing
#define motzkin 1             // parameter determining the relaxation method used: 1, 0 for Motzkin=1 or MinOver, respectively. Choose Motzkin for bad chemical potentials priors,  Minover if the priors are good and you need to use the chemical potentials.
#define cycle_correction  1   // if cycle_correction=1 minimal correction (the minimum is set to 0, no sign change)   else local minimization of the L2 norm  (faster convergence, a priori reversibility assignment could be violated)
#define lambda 1.9            // lenght of the relaxation step  (use lambda=1.9 if motzkin=1, lambda=0.0001 if motzkin=0)
#define steplenght 0.01       // annealing rate
#define numberann 100         // max number of annealing cooling cycles
#define epsilon 0.000000001   // Minimum value of flux consiedered signficant: i.e., if |flux|<epsilon  flux=0


// VARIABLES DECLARATION
int unsat[max_LUC];                   // list of l.u.c.
int nofound=0;                        // counting the number of time before restarting chemical potentials
int ok;                               // ok==1: thermodynamics constraints satisfied 
std::vector<int> metab[M];            // declaring pointers to reactions/metabolites and stoichiometry
std::vector<int> react[N];                                  
std::vector<double> metab2[M];
std::vector<double> react2[N];
int check[N];                         // constraints that have to be satisfied
double flux[N];                       // fluxes
double chempot[M];                    // chemical potentials
fstream cycle_file;

// FUNCTIONS
double casual(){
    int m = random();
    double r = double(m)/(RAND_MAX+1.0);                      // random number generator between 0 and 1
    return r;
}

void read_input(){
    fstream input_file;
    input_file.open(Stoichiometric_matrix_filename, ios::in);                    // read the stoichiometric matrix   rows: metabolites   columns: reactions 
    int i1=0;
    int j1=0;        
    double pepe;
    while(input_file >> pepe){
	if(pepe!=0){
	    metab[i1].push_back(j1);
	    metab2[i1].push_back(pepe);
	    react[j1].push_back(i1);
	    react2[j1].push_back(pepe);
	}
	j1++;
	if(j1==N){
	    j1=0;
	    i1++;
	}
    }        
    input_file.close();  
    for(int i=0;i<=N-1;i++){
	check[i]=1;                                                     // excluding uptakes  and  effective reactions like biomass
	if(react[i].size()==1) check[i]=0;
	else  for(int j=0;j<=react[i].size()-1;j++) if( react2[i][j]!=int(react2[i][j]) ) check[i]=0;
    }
    input_file.open(fluxes_filename,ios::in);
    for(int i=0;i<=N-1;i++){
	input_file >>  flux[i];           // reading from fluxes.dat a flux configuration
	if(fabs(flux[i])<epsilon) flux[i]=0;
    }
    input_file.close();
    if(motzkin==0){
	input_file.open(potentials_filename,ios::in);
	for(int i=0;i<=M-1;i++) input_file >>  flux[i];           // reading from potentials.dat a chemical potential vector
	input_file.close();
    }
    else for(int i=0;i<=M-1;i++) chempot[i] = 0.5-casual();  // random assignment of chemical potentials
}

void write_output(){ 
    fstream output_file;

    output_file.open(feasible_fluxes_filename, ios::out);
    output_file << "flux vector:"<< endl; 
    for(int i=0;i<=N-1;i++)  output_file << flux[i] << endl;             
    output_file.close();

    output_file.open(feasible_potentials_filename, ios::out);                                  
    output_file << "chemical potential vector:"<< endl; 
    for(int i=0;i<=M-1;i++)  output_file << chempot[i] << endl;   
    output_file.close();                    
}

void Montecarlo(){
    // preprocessing
    std::vector <int> unique;
    unique.push_back(unsat[0]);
    for(int i=1;i<=max_LUC-1;i++){
	int oko=0;
	for(int j=0;j<=i-1;j++) if(unsat[i]==unsat[j])oko++;     // choose distinct l.u.c. from unsat
	if(oko==0) unique.push_back(unsat[i]);
    }
    int present[unique.size()];
    for(int i=0;i<=unique.size()-1;i++) present[i]=1;
    int change;
    do{                                                                                // excluding leaves
	change=0;
	for(int i=0;i<=unique.size()-1;i++){
	    if(present[i]==1){
		int r = unique[i];
		int match=1;
		for(int j=0;j<=react[r].size()-1;j++){
		    int k=0;
		    int okei=0;
		    do{
			if(k!=i && present[k]==1){
			    int r1=unique[k];
			    for(int j1=0;j1<=react[r1].size()-1;j1++) if(react[r][j]==react[r1][j1]) okei=1;
			}
			k++;
		    }while(k<unique.size() && okei==0);
		    if(okei==0) match=0;
		}


		if(match==0){
		    present[i]=0;
		    change=1;
		} 
	    }
	}

    }while(change==1);
    std::vector<int> matched;                                                                      // in matched there are only reactions that are not leaves in unsat
    for(int i=0;i<=unique.size()-1;i++) if(present[i]==1) matched.push_back(unique[i]);             
    unique.clear();
    nofound++;
    if(nofound==max_MC_sweeps || matched.size()==0){                                    // if matched is empty or the MC doesn't find any cycles after max_MC_sweeps trials, reassign the potentials
	for(int i=0;i<=M-1;i++) chempot[i] = (0.5-casual());  
	nofound=0;
    }
    if(matched.size()>1){                       // MC start
	int dim = matched.size();            
	int J[dim][dim];                    // interactions
	int num[dim];                       // variables
	int numtot=0;                         // sum of num
	for(int i=0;i<=dim-1;i++){
	    num[i]=int(2*casual());           // random start, calculate sum of num
	    numtot += num[i];
	}
	for(int i=0;i<=dim-1;i++){                                                             // interactions assignment
	    for(int j=i;j<=dim-1;j++){
		J[i][j]=0;
		for(int po=0;po<=react[matched[i]].size()-1;po++) for(int pu=0;pu<=react[matched[j]].size()-1;pu++) 
		    if(react[matched[i]][po]==react[matched[j]][pu]){
			int s=1;
			if(flux[matched[i]]*flux[matched[j]]<0) s=-1;
			J[i][j] += s*react2[matched[i]][po]*react2[matched[j]][pu];
		    }
		J[j][i]=J[i][j];
	    }
	} 
	double H=0;
	for(int i=0;i<=dim-1;i++) for(int j=0;j<=dim-1;j++) H += J[i][j]*num[i]*num[j];     // calculate energy      
	int refresh=0;
	int end_MC=0;
	int step=-1;
	do{  

	    int en=int(casual()*dim);
	    int p=1;
	    if(num[en]>0 && casual()<0.5) p=-1;
	    double delta = 0; 
	    for(int i=0;i<=dim-1;i++) delta += J[i][en]*num[i];
	    delta = 2*p*delta + J[en][en];
	    if(delta<0 || casual()<exp(-delta/T1)){                         // metropolis rule
		num[en] += p;
		numtot += p;
		H += delta;
	    }
	    T1+=step*steplenght/double(matched.size());
	    if(T1<=0){                           // cooling
		refresh++;
		step=1;
	    } 
	    if(T1>=1.5) step=-1;                     // annealing
	    if( (H==0 && numtot>0) || refresh>numberann) end_MC=1;
	}while(end_MC==0);                                               // the MC stops if a cycle is found or after numberann coolings

	if(refresh<=numberann && numtot>0){                   // if TRUE a cycle has been found
	    nofound=0;
	    int many=0;
	    for(int i=0;i<=matched.size()-1;i++) if(num[i]>0) many++;               
	    cycle_file << "cycle of lenght  "  << many << endl;
	    if(cycle_correction==1){
		double minimum=100000;
		for(int i=0;i<=matched.size()-1;i++){
		    if(num[i]>0){
			int l = matched[i];            // print the cycle: reaction index, value in the cycle, current value of the flux
			cycle_file << l  << "  " << num[i] << "  "  << flux[l] << endl;

			if(fabs(flux[l])/double(num[i])<minimum) minimum=fabs(flux[l])/double(num[i]);
		    }
		}
		for(int o=0;o<=matched.size()-1;o++){
		    if(num[o]>0){
			int es=1;                                           // cycle minimal correction (set to 0 the minimum, no sign change)
			if(flux[matched[o]]<0) es=-1;
			flux[matched[o]]-=es*num[o]*minimum;
			if(fabs(flux[matched[o]])<epsilon) flux[matched[o]]=0;
		    }
		}
	    }
	    else{
		double numer=0;
		double denom=0;
		for(int i=0;i<=matched.size()-1;i++){
		    if(num[i]>0){
			int l = matched[i];
			cycle_file << l  << "  " << num[i] << "  "  << flux[l] << endl;   // cycle correction by locally minimizing L2 norm (faster)
			numer+=fabs(flux[l])*num[i];
			denom+=num[i]*num[i];
		    }
		}
		double L=numer/denom;
		for(int o=0;o<=matched.size()-1;o++){
		    if(num[o]>0){
			int es=1;
			if(flux[matched[o]]<0) es=-1;
			flux[matched[o]]-=es*num[o]*L;
			if(fabs(flux[matched[o]])<epsilon) flux[matched[o]]=0;
		    }
		}
	    }
	}
    }

}


void Relaxation(int w, int count, int min, double cmin){
    for(int j=0;j<=N-1;j++){
	if(fabs(flux[j])>w  && check[j]==1){                                                                                // calculate the l.u.c.
	    double freen=0;
	    for(int i=0;i<=react[j].size()-1;i++) freen += react2[j][i]*chempot[react[j][i]];
	    if(flux[j]>0) freen = -freen;
	    if(freen<cmin){  
		min = j;
		cmin = freen;
	    }
	}
    }
    unsat[count]=min;    // list of l.u.c. update
    if(cmin>=0) ok=1;    // if TRUE constraints satisfied                 
    else{
	double alpha;

	if(motzkin==0) alpha=lambda;                                       // MinOver relaxation  
	else{
	    alpha=-lambda*cmin;                                // Motzkin relaxation
	    double fact=0;
	    for(int j=0;j<=react[min].size()-1;j++) fact+=react2[min][j]*react2[min][j];                
	    alpha/=fact;
	    if(alpha<epsilon) alpha=epsilon;
	}                        
	int s=1;
	if(flux[min]>0) s=-1;
	for(int j=0;j<=react[min].size()-1;j++) chempot[react[min][j]] += s*alpha*react2[min][j];              // chemical potentials update
    }
}



// MAIN FUNCTION
int main(){
    srand(time(0));                             // seed for random numbers
    read_input();                               // read input
    cycle_file.open(cycles_filename, ios::out);
    for(int w=1000;w>=0;w--){                   // acting sequentially on the subset of fluxes:  |flux|>w
	ok=0;
	int count=0;                            
	double cmin=10000;                                                         
	int min=0;
	do{  
	    if(count==max_LUC*max2){
		Montecarlo();               
		count=0;
	    }
	    Relaxation(w, count%max_LUC, min, cmin);
	    count++;
	}while(ok==0);
    }
    cycle_file.close();
    write_output();                       // write output
    cout << "machine time: "<< double(clock())/double(CLOCKS_PER_SEC) << "s" << endl;           // print on standard output the machine time (s)                  

    return 0;
}


